// Package main - File writing functionality for schemagen.
// This module handles writing generated Go code to files with proper formatting.
package main

import (
	"bytes"
	"fmt"
	"go/format"
	"os"
	"path/filepath"
	"strings"
	"text/template"

	"github.com/kaptinlin/jsonschema"
)

// FileWriter handles writing generated Go code to files
type FileWriter struct {
	outputSuffix string
	packageName  string
	dryRun       bool
	verbose      bool
	templates    *template.Template
}

// TemplateData holds data for code generation templates
type TemplateData struct {
	Package string
	Imports []string
	Methods []MethodData
}

// MethodData holds data for a single Schema() method
type MethodData struct {
	Receiver       string // e.g., "s TestUser"
	StructName     string // e.g., "TestUser"
	Properties     []string
	RequiredFields string    // comma-separated quoted field names
	Definitions    []DefData // $defs for referenced schemas
}

// DefData holds data for a single $defs entry
type DefData struct {
	Name       string   // Definition name
	Properties []string // Properties of the definition
	Required   string   // Required fields for the definition
}

// PropertyData holds data for a single property
type PropertyData struct {
	JSONName   string
	BaseSchema string
	Validators []string
}

// Template definitions
const schemaFileTemplate = `// Code generated by schemagen. DO NOT EDIT.

package {{.Package}}

import (
{{range .Imports}}	"{{.}}"
{{end}})

{{range .Methods}}
func ({{.Receiver}}) Schema() *jsonschema.Schema {
	{{if .Definitions}}return jsonschema.Object(
{{range .Properties}}		{{.}},
{{end}}{{if .RequiredFields}}		jsonschema.Required({{.RequiredFields}}),
{{end}}		jsonschema.Defs(map[string]*jsonschema.Schema{
{{range .Definitions}}			"{{.Name}}": jsonschema.Object(
{{range .Properties}}				{{.}},
{{end}}{{if .Required}}				jsonschema.Required({{.Required}}),
{{end}}			),
{{end}}		}),
	){{else}}return jsonschema.Object(
{{range .Properties}}		{{.}},
{{end}}{{if .RequiredFields}}		jsonschema.Required({{.RequiredFields}}),
{{end}}	){{end}}
}
{{end}}`

// NewFileWriter creates a new file writer instance
func NewFileWriter(outputSuffix, packageName string, dryRun, verbose bool) (*FileWriter, error) {
	if outputSuffix == "" {
		outputSuffix = "_schema.go"
	}

	// Parse and initialize templates
	templates, err := template.New("schema").Parse(schemaFileTemplate)
	if err != nil {
		return nil, fmt.Errorf("%w: %w", jsonschema.ErrTemplateParsing, err)
	}

	return &FileWriter{
		outputSuffix: outputSuffix,
		packageName:  packageName,
		dryRun:       dryRun,
		verbose:      verbose,
		templates:    templates,
	}, nil
}

// WriteGeneratedCode writes the generated schema code for a struct using templates
func (w *FileWriter) WriteGeneratedCode(filePath, packageName string, methods []MethodData) error {
	// Generate output file name
	outputFile := w.generateOutputFileName(filePath)

	// Prepare template data
	templateData := TemplateData{
		Package: w.getPackageName(packageName),
		Imports: []string{"github.com/kaptinlin/jsonschema"},
		Methods: methods,
	}

	// Execute template
	var buf bytes.Buffer
	err := w.templates.Execute(&buf, templateData)
	if err != nil {
		return fmt.Errorf("%w: %w", jsonschema.ErrTemplateExecution, err)
	}

	// Format the generated code
	formattedCode, err := w.formatCode(buf.String())
	if err != nil {
		return fmt.Errorf("%w: %w", jsonschema.ErrCodeFormatting, err)
	}

	if w.dryRun {
		// In dry-run mode, just print the code
		fmt.Printf("// Generated code for %s (dry-run mode)\n", outputFile)
		fmt.Printf("// File: %s\n", outputFile)
		fmt.Println(formattedCode)
		return nil
	}

	// Write to file
	err = w.writeToFile(outputFile, formattedCode)
	if err != nil {
		return fmt.Errorf("%w: %w", jsonschema.ErrFileWrite, err)
	}

	if w.verbose {
		fmt.Printf("Generated file: %s\n", outputFile)
	}

	return nil
}

// getPackageName returns the package name to use, with fallback logic
func (w *FileWriter) getPackageName(defaultPackage string) string {
	if w.packageName != "" {
		return w.packageName
	}
	return defaultPackage
}

// generateOutputFileName generates the output file name based on input file
func (w *FileWriter) generateOutputFileName(inputFile string) string {
	dir := filepath.Dir(inputFile)
	base := filepath.Base(inputFile)

	// Remove .go extension and add suffix
	base = strings.TrimSuffix(base, ".go")

	return filepath.Join(dir, base+w.outputSuffix)
}

// formatCode formats Go code using go/format
func (w *FileWriter) formatCode(code string) (string, error) {
	formatted, err := format.Source([]byte(code))
	if err != nil {
		// If formatting fails, return original code with error
		return code, fmt.Errorf("%w: %w", jsonschema.ErrCodeFormatting, err)
	}
	return string(formatted), nil
}

// writeToFile writes content to a file
func (w *FileWriter) writeToFile(filename, content string) error {
	// Ensure directory exists
	dir := filepath.Dir(filename)
	if err := os.MkdirAll(dir, 0750); err != nil {
		return fmt.Errorf("%w: %s: %w", jsonschema.ErrDirectoryCreation, dir, err)
	}

	// Validate filename path to prevent directory traversal attacks
	// Only allow relative paths and ensure no path traversal
	cleanPath := filepath.Clean(filename)
	if filepath.IsAbs(cleanPath) || strings.Contains(cleanPath, "..") {
		return fmt.Errorf("%w: %s", jsonschema.ErrInvalidFilenamePath, filename)
	}

	// Ensure the cleaned path is within the current working directory
	absPath, err := filepath.Abs(cleanPath)
	if err != nil {
		return fmt.Errorf("%w: %w", jsonschema.ErrAbsolutePathResolution, err)
	}

	currentDir, err := os.Getwd()
	if err != nil {
		return fmt.Errorf("%w: %w", jsonschema.ErrCurrentDirectoryAccess, err)
	}

	if !strings.HasPrefix(absPath, currentDir) {
		return fmt.Errorf("%w: %s", jsonschema.ErrPathOutsideDirectory, filename)
	}

	// Write file
	file, err := os.Create(cleanPath)
	if err != nil {
		return fmt.Errorf("%w: %s: %w", jsonschema.ErrFileCreation, cleanPath, err)
	}
	defer func() {
		if closeErr := file.Close(); closeErr != nil {
			// Log the error but don't return it since we're in defer
			if w.verbose {
				fmt.Printf("Warning: failed to close file %s: %v\n", filename, closeErr)
			}
		}
	}()

	_, err = file.WriteString(content)
	if err != nil {
		return fmt.Errorf("%w: %s: %w", jsonschema.ErrContentWrite, filename, err)
	}

	return nil
}
