package resolve

import (
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"

	"github.com/kaptinlin/messageformat-go/pkg/datamodel"
	"github.com/kaptinlin/messageformat-go/pkg/functions"
	"github.com/kaptinlin/messageformat-go/pkg/messagevalue"
)

func TestGetValue(t *testing.T) {
	scope := map[string]interface{}{
		"name":    "Alice",
		"age":     30,
		"user.id": 123,
		"settings": map[string]interface{}{
			"theme": "dark",
			"lang":  "en",
		},
	}

	// Direct lookup
	assert.Equal(t, "Alice", getValue(scope, "name"))
	assert.Equal(t, 30, getValue(scope, "age"))

	// Dotted property access
	assert.Equal(t, "dark", getValue(scope, "settings.theme"))
	assert.Equal(t, "en", getValue(scope, "settings.lang"))

	// Non-existent key
	assert.Nil(t, getValue(scope, "nonexistent"))

	// Non-scope value
	assert.Nil(t, getValue("not a scope", "name"))
	assert.Nil(t, getValue(nil, "name"))
}

func TestLookupVariableRef(t *testing.T) {
	ctx := NewContext(
		[]string{"en"},
		map[string]functions.MessageFunction{
			"string": functions.StringFunction,
		},
		map[string]interface{}{
			"name": "Alice",
			"age":  30,
		},
		nil,
	)

	// Existing variable
	ref := datamodel.NewVariableRef("name")
	value := lookupVariableRef(ctx, ref)
	assert.Equal(t, "Alice", value)

	// Non-existing variable
	ref2 := datamodel.NewVariableRef("nonexistent")
	value2 := lookupVariableRef(ctx, ref2)
	assert.Nil(t, value2)
}

func TestResolveVariableRef(t *testing.T) {
	ctx := NewContext(
		[]string{"en"},
		map[string]functions.MessageFunction{
			"string": functions.StringFunction,
			"number": functions.NumberFunction,
		},
		map[string]interface{}{
			"name": "Alice",
			"age":  30,
		},
		nil,
	)

	// String variable
	ref := datamodel.NewVariableRef("name")
	result := ResolveVariableRef(ctx, ref)
	assert.Equal(t, "string", result.Type())

	// Number variable
	ref2 := datamodel.NewVariableRef("age")
	result2 := ResolveVariableRef(ctx, ref2)
	assert.Equal(t, "number", result2.Type())

	// Non-existing variable (should return fallback)
	ref3 := datamodel.NewVariableRef("nonexistent")
	result3 := ResolveVariableRef(ctx, ref3)
	assert.Equal(t, "fallback", result3.Type())
}

func TestUnresolvedExpression(t *testing.T) {
	expr := datamodel.NewExpression(
		datamodel.NewLiteral("test"),
		nil,
		nil,
	)
	scope := map[string]interface{}{"key": "value"}

	unresolved := NewUnresolvedExpression(expr, scope)

	assert.Equal(t, expr, unresolved.Expression)
	assert.Equal(t, scope, unresolved.Scope)
}

func TestIsScope(t *testing.T) {
	// Valid scopes
	assert.True(t, isScope(map[string]interface{}{}))
	assert.True(t, isScope(map[interface{}]interface{}{}))
	assert.True(t, isScope(struct{}{}))

	// Invalid scopes
	assert.False(t, isScope(nil))
	assert.False(t, isScope("string"))
	assert.False(t, isScope(123))
	assert.False(t, isScope([]string{}))
}

func TestGetFirstLocale(t *testing.T) {
	// With locales
	assert.Equal(t, "en", getFirstLocale([]string{"en", "fr"}))
	assert.Equal(t, "fr", getFirstLocale([]string{"fr"}))

	// Without locales
	assert.Equal(t, "en", getFirstLocale([]string{}))
	assert.Equal(t, "en", getFirstLocale(nil))
}

// TestVariables tests variable resolution with different value types
// TypeScript original code:
//
//	describe('variables', () => {
//	  let mf: MessageFormat;
//	  beforeEach(() => {
//	    mf = new MessageFormat('en', '{$val}');
//	  });
func TestVariables(t *testing.T) {
	// Helper function to create a variable expression and resolve it
	resolveVariable := func(t *testing.T, value interface{}) messagevalue.MessageValue {
		// Create a variable reference expression
		varRef := datamodel.NewVariableRef("val")

		// Create context with the value
		ctx := NewContext(
			[]string{"en"},
			functions.DefaultFunctions,
			map[string]interface{}{
				"val": value,
			},
			nil,
		)

		// Resolve the variable
		return ResolveVariableRef(ctx, varRef)
	}

	// Helper function to format variable to parts
	formatToParts := func(t *testing.T, value interface{}) []messagevalue.MessagePart {
		mv := resolveVariable(t, value)
		parts, err := mv.ToParts()
		require.NoError(t, err)
		return parts
	}

	// Helper function to format variable to string
	formatToString := func(t *testing.T, value interface{}) string {
		mv := resolveVariable(t, value)
		str, err := mv.ToString()
		require.NoError(t, err)
		return str
	}

	t.Run("number", func(t *testing.T) {
		// TypeScript original code:
		//   expect(mf.format({ val: 42 })).toBe('42');
		//   expect(mf.formatToParts({ val: 42 })).toEqual([
		//     {
		//       type: 'number',
		//       dir: 'ltr',
		//       locale: 'en',
		//       parts: [{ type: 'integer', value: '42' }]
		//     }
		//   ]);

		assert.Equal(t, "42", formatToString(t, 42))

		parts := formatToParts(t, 42)
		require.Len(t, parts, 1)
		assert.Equal(t, "number", parts[0].Type())
		assert.Equal(t, "42", parts[0].Value())
		assert.Equal(t, "en", parts[0].Locale())
		// Note: Go implementation may not have the exact same nested parts structure as TypeScript
	})

	t.Run("bigint", func(t *testing.T) {
		// TypeScript original code:
		//   const val = BigInt(42);
		//   expect(mf.format({ val })).toBe('42');
		//   expect(mf.formatToParts({ val })).toEqual([
		//     {
		//       type: 'number',
		//       dir: 'ltr',
		//       locale: 'en',
		//       parts: [{ type: 'integer', value: '42' }]
		//     }
		//   ]);

		// In Go, we use int64 to represent big integers
		val := int64(42)
		assert.Equal(t, "42", formatToString(t, val))

		parts := formatToParts(t, val)
		require.Len(t, parts, 1)
		assert.Equal(t, "number", parts[0].Type())
		assert.Equal(t, "42", parts[0].Value())
		assert.Equal(t, "en", parts[0].Locale())
	})

	t.Run("float", func(t *testing.T) {
		// TypeScript original code uses Number object, we test with float64
		val := 42.0
		assert.Equal(t, "42", formatToString(t, val))

		parts := formatToParts(t, val)
		require.Len(t, parts, 1)
		assert.Equal(t, "number", parts[0].Type())
		assert.Equal(t, "42", parts[0].Value())
		assert.Equal(t, "en", parts[0].Locale())
	})

	t.Run("wrapped number", func(t *testing.T) {
		// TypeScript original code:
		//   const val = { valueOf: () => BigInt(42) };
		//   expect(mf.formatToParts({ val })).toEqual([
		//     { type: 'bidiIsolation', value: '\u2068' },
		//     { type: 'unknown', value: val },
		//     { type: 'bidiIsolation', value: '\u2069' }
		//   ]);

		// In Go, we can't easily replicate the valueOf behavior,
		// but we can test with a complex object that doesn't have a direct number conversion
		type customValue struct {
			value int64
		}
		val := customValue{value: 42}

		parts := formatToParts(t, val)
		// The exact behavior may differ, but we expect it to be handled as an unknown type
		// with bidi isolation
		assert.True(t, len(parts) >= 1)
		// Check if bidi isolation is applied for unknown types
		if len(parts) == 3 {
			assert.Equal(t, "bidiIsolation", parts[0].Type())
			assert.Equal(t, "\u2068", parts[0].Value())
			assert.Equal(t, "bidiIsolation", parts[2].Type())
			assert.Equal(t, "\u2069", parts[2].Value())
		}
	})

	t.Run("number with options", func(t *testing.T) {
		// TypeScript original code:
		//   const val = Object.assign(new Number(42), {
		//     options: { minimumFractionDigits: 1 }
		//   });
		//   expect(mf.format({ val })).toBe('42.0');

		// In Go, we can simulate this by creating a NumberValue with options directly
		// This tests the number formatting with specific options

		// Create a NumberValue with options
		options := map[string]interface{}{
			"minimumFractionDigits": 1,
		}
		nv := messagevalue.NewNumberValue(42, "en", "test", options)

		str, err := nv.ToString()
		require.NoError(t, err)
		assert.Equal(t, "42.0", str)

		parts, err := nv.ToParts()
		require.NoError(t, err)
		require.Len(t, parts, 1)
		assert.Equal(t, "number", parts[0].Type())
		// The formatted value should include the decimal
		assert.Contains(t, parts[0].Value(), "42")
		assert.Contains(t, parts[0].Value(), ".")
	})
}

// TestVariablePaths tests variable path resolution
// TypeScript original code:
//
//	describe('Variable paths', () => {
//	  let mf: MessageFormat;
//	  beforeEach(() => {
//	    mf = new MessageFormat('en', '{$user.name}');
//	  });
func TestVariablePaths(t *testing.T) {
	// Helper function to resolve a variable path
	resolveVariablePath := func(t *testing.T, varName string, values map[string]interface{}) []messagevalue.MessagePart {
		// Create a variable reference expression
		varRef := datamodel.NewVariableRef(varName)

		// Create context with the values
		ctx := NewContext(
			[]string{"en"},
			functions.DefaultFunctions,
			values,
			nil,
		)

		// Resolve the variable
		mv := ResolveVariableRef(ctx, varRef)
		parts, err := mv.ToParts()
		require.NoError(t, err)
		return parts
	}

	t.Run("top-level match", func(t *testing.T) {
		// TypeScript original code:
		//   expect(mf.formatToParts({ 'user.name': 42 })).toEqual([
		//     {
		//       type: 'number',
		//       dir: 'ltr',
		//       locale: 'en',
		//       parts: [{ type: 'integer', value: '42' }]
		//     }
		//   ]);

		values := map[string]interface{}{
			"user.name": 42,
		}

		parts := resolveVariablePath(t, "user.name", values)
		require.Len(t, parts, 1)
		assert.Equal(t, "number", parts[0].Type())
		assert.Equal(t, "42", parts[0].Value())
		assert.Equal(t, "en", parts[0].Locale())
	})

	t.Run("scoped match", func(t *testing.T) {
		// TypeScript original code:
		//   expect(mf.formatToParts({ user: { name: 42 } })).toEqual([
		//     {
		//       type: 'number',
		//       dir: 'ltr',
		//       locale: 'en',
		//       parts: [{ type: 'integer', value: '42' }]
		//     }
		//   ]);

		values := map[string]interface{}{
			"user": map[string]interface{}{
				"name": 42,
			},
		}

		parts := resolveVariablePath(t, "user.name", values)
		require.Len(t, parts, 1)
		assert.Equal(t, "number", parts[0].Type())
		assert.Equal(t, "42", parts[0].Value())
		assert.Equal(t, "en", parts[0].Locale())
	})

	t.Run("top-level overrides scoped match", func(t *testing.T) {
		// TypeScript original code:
		//   expect(mf.formatToParts({ user: { name: 13 }, 'user.name': 42 })).toEqual([
		//     {
		//       type: 'number',
		//       dir: 'ltr',
		//       locale: 'en',
		//       parts: [{ type: 'integer', value: '42' }]
		//     }
		//   ]);

		values := map[string]interface{}{
			"user": map[string]interface{}{
				"name": 13,
			},
			"user.name": 42,
		}

		parts := resolveVariablePath(t, "user.name", values)
		require.Len(t, parts, 1)
		assert.Equal(t, "number", parts[0].Type())
		assert.Equal(t, "42", parts[0].Value()) // Should be 42, not 13
		assert.Equal(t, "en", parts[0].Locale())
	})
}
