package cmd

import (
	"maps"
	"slices"

	"github.com/spf13/cobra"

	"github.com/evilmartians/lefthook/internal/command"
	"github.com/evilmartians/lefthook/internal/config"
	"github.com/evilmartians/lefthook/internal/log"
)

type run struct{}

func (run) New(opts *command.Options) *cobra.Command {
	runArgs := command.RunArgs{}

	runHookCompletions := func(cmd *cobra.Command, args []string, toComplete string) (ret []string, compDir cobra.ShellCompDirective) {
		compDir = cobra.ShellCompDirectiveNoFileComp
		if len(args) != 0 {
			return ret, compDir
		}
		ret = command.ConfigHookCompletions(opts)
		other := slices.Sorted(maps.Keys(config.AvailableHooks))
		ret = append(ret, other...)
		return ret, compDir
	}

	runHookCommandCompletions := func(cmd *cobra.Command, args []string, toComplete string) (ret []string, compDir cobra.ShellCompDirective) {
		compDir = cobra.ShellCompDirectiveNoFileComp
		if len(args) == 0 {
			return ret, compDir
		}
		ret = command.ConfigHookCommandCompletions(opts, args[0])
		return ret, compDir
	}

	runHookJobCompletions := func(cmd *cobra.Command, args []string, toComplete string) (ret []string, compDir cobra.ShellCompDirective) {
		compDir = cobra.ShellCompDirectiveNoFileComp
		if len(args) == 0 {
			return ret, compDir
		}
		ret = command.ConfigHookJobCompletions(opts, args[0])
		return ret, compDir
	}

	runCmd := cobra.Command{
		Use:               "run hook-name [git args...]",
		Short:             "Execute group of hooks",
		Example:           "lefthook run pre-commit",
		ValidArgsFunction: runHookCompletions,
		Args:              cobra.MinimumNArgs(1),
		RunE: func(cmd *cobra.Command, args []string) error {
			// args[0] - hook name
			// args[1:] - git hook arguments, number and value depends on the hook
			return command.Run(opts, runArgs, args[0], args[1:])
		},
	}

	runCmd.Flags().BoolVarP(
		&runArgs.Force, "force", "f", false,
		"force execution of commands that can be skipped",
	)

	runCmd.Flags().BoolVarP(
		&runArgs.NoTTY, "no-tty", "n", false,
		"run hook non-interactively, disable spinner",
	)

	runCmd.Flags().BoolVar(
		&runArgs.FailOnChanges, "fail-on-changes", false,
		"fail hook if it modifies any file",
	)

	runCmd.Flags().BoolVar(
		&runArgs.AllFiles, "all-files", false,
		"run hooks on all files",
	)

	runCmd.Flags().BoolVar(
		&runArgs.NoAutoInstall, "no-auto-install", false,
		"skip updating git hooks",
	)

	runCmd.Flags().BoolVar(
		&runArgs.SkipLFS, "skip-lfs", false,
		"skip running git lfs",
	)

	runCmd.Flags().BoolVar(
		&runArgs.FilesFromStdin, "files-from-stdin", false,
		"get files from standard input, null-separated",
	)

	runCmd.Flags().StringSliceVar(
		&runArgs.Files, "files", nil,
		"run on specified files, comma-separated",
	)

	runCmd.Flags().StringArrayVar(
		&runArgs.Files, "file", nil,
		"run on specified file (repeat for multiple files). takes precedence over --all-files",
	)

	runCmd.Flags().StringArrayVar(
		&runArgs.Exclude, "exclude", nil,
		"exclude specified file (repeat for multiple files)",
	)

	runCmd.Flags().StringSliceVar(
		&runArgs.RunOnlyCommands, "commands", nil,
		"run only specified commands",
	)

	runCmd.Flags().StringSliceVar(
		&runArgs.RunOnlyJobs, "jobs", nil,
		"run only specified jobs",
	)

	runCmd.Flags().StringSliceVar(
		&runArgs.RunOnlyTags, "tags", nil,
		"run only jobs with specified tags",
	)

	runCmd.Flags().BoolVar(
		&runArgs.NoStageFixed, "no-stage-fixed", false,
		"disable 'stage_fixed: true' setting",
	)

	err := runCmd.Flags().MarkDeprecated("files", "use --file flag instead")
	if err != nil {
		log.Warn("Unexpected error:", err)
	}

	_ = runCmd.RegisterFlagCompletionFunc("commands", runHookCommandCompletions)

	_ = runCmd.RegisterFlagCompletionFunc("jobs", runHookJobCompletions)

	return &runCmd
}
