package flvio

import (
	"testing"
)

func TestDecodeInteger(t *testing.T) {
	tests := []struct {
		value int
		bytes []byte
	}{
		{0x0, []byte{0x04, 0x00}},
		{0x1, []byte{0x04, 0x01}},
		{0x7f, []byte{0x04, 0x7f}},
		{0x80, []byte{0x04, 0x81, 0x00}},
		{0x81, []byte{0x04, 0x81, 0x01}},
		{0x3fff, []byte{0x04, 0xff, 0x7f}},
		{0x4000, []byte{0x04, 0x81, 0x80, 0x00}},
		{0x1fffff, []byte{0x04, 0xff, 0xff, 0x7f}},
		{0x200000, []byte{0x04, 0x80, 0xc0, 0x80, 0x00}},
		{0x1fffffff, []byte{0x04, 0xff, 0xff, 0xff, 0xff}},
		{0x200000, []byte{0x04, 0x80, 0xc0, 0x80, 0x00}},
		{1<<28 - 1, []byte{0x04, 0xbf, 0xff, 0xff, 0xff}},
		{1 << 28, []byte{0x04, 0xc0, 0x80, 0x80, 0x00}},
	}

	for _, test := range tests {
		n := 0
		val, err := ParseAMF3Val(test.bytes, &n)
		assertEqual(t, err, nil, "reading U29 %x", test.value)
		assertEqual(t, val.(int), test.value, "reading U29 %x", test.value)
	}
}

func TestAmf3Decoder(t *testing.T) {
	data := []byte{
		// 0x11, // amf3
		0x0a, // object

		0x81, 0x13, // traits, not dynamic, 9 count
		0x4f, 0x66, 0x6c, 0x65, 0x78, 0x2e, 0x6d, 0x65, 0x73, 0x73, 0x61, 0x67, 0x69, 0x6e, 0x67, 0x2e, 0x6d, 0x65, 0x73, 0x73, 0x61, 0x67, 0x65, 0x73, 0x2e, 0x52, 0x65, 0x6d, 0x6f, 0x74, 0x69, 0x6e, 0x67, 0x4d, 0x65, 0x73, 0x73, 0x61, 0x67, 0x65, // class name: flex.messaging.messages.RemotingMessage
		0x0d, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, // source
		0x13, 0x6f, 0x70, 0x65, 0x72, 0x61, 0x74, 0x69, 0x6f, 0x6e, // operation
		0x0f, 0x68, 0x65, 0x61, 0x64, 0x65, 0x72, 0x73, // headers
		0x17, 0x64, 0x65, 0x73, 0x74, 0x69, 0x6e, 0x61, 0x74, 0x69, 0x6f, 0x6e, // destination
		0x13, 0x6d, 0x65, 0x73, 0x73, 0x61, 0x67, 0x65, 0x49, 0x64, // messageId
		0x11, 0x63, 0x6c, 0x69, 0x65, 0x6e, 0x74, 0x49, 0x64, // clientId
		0x15, 0x74, 0x69, 0x6d, 0x65, 0x54, 0x6f, 0x4c, 0x69, 0x76, 0x65, // timeToLive
		0x13, 0x74, 0x69, 0x6d, 0x65, 0x73, 0x74, 0x61, 0x6d, 0x70, // timestamp
		0x09, 0x62, 0x6f, 0x64, 0x79, // body

		0x01, // null

		0x06, 0x29, // string, len 20
		0x6c, 0x69, 0x73, 0x74, 0x41, 0x6c, 0x6c, 0x50, 0x72, 0x61, 0x63, 0x74, 0x69, 0x63, 0x65, 0x47, 0x61, 0x6d, 0x65, 0x73, // listAllPracticeGames

		0x0a,       // object
		0x0b, 0x01, // dynamic
		// key: DSRequestTimeout
		0x21, 0x44, 0x53, 0x52, 0x65, 0x71, 0x75, 0x65, 0x73, 0x74, 0x54, 0x69, 0x6d, 0x65, 0x6f, 0x75, 0x74,
		// value: integer 60
		0x04, 0x3c,
		// key: DSEndpoint
		0x15, 0x44, 0x53, 0x45, 0x6e, 0x64, 0x70, 0x6f, 0x69, 0x6e, 0x74,
		// value: string my-rtmps
		0x06, 0x11, 0x6d, 0x79, 0x2d, 0x72, 0x74, 0x6d, 0x70, 0x73,
		// key: DSId
		0x09, 0x44, 0x53, 0x49, 0x64,
		// value: string BE241234-1E19-03C3-904C-2163660AB47F
		0x06, 0x49, 0x42, 0x45, 0x32, 0x34, 0x31, 0x32, 0x33, 0x34, 0x2d, 0x31, 0x45, 0x31, 0x39, 0x2d, 0x30, 0x33, 0x43, 0x33, 0x2d, 0x39, 0x30, 0x34, 0x43, 0x2d, 0x32, 0x31, 0x36, 0x33, 0x36, 0x36, 0x30, 0x41, 0x42, 0x34, 0x37, 0x46,
		// end
		0x01,

		0x06, 0x17, // string, len 11
		0x67, 0x61, 0x6d, 0x65, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, // gameService

		0x06, 0x49, // string, len 36
		0x31, 0x34, 0x43, 0x30, 0x46, 0x42, 0x38, 0x30, 0x2d, 0x46, 0x37, 0x43, 0x34, 0x2d, 0x42, 0x42, 0x34, 0x42, 0x2d, 0x31, 0x46, 0x32, 0x34, 0x2d, 0x34, 0x39, 0x36, 0x32, 0x38, 0x30, 0x42, 0x38, 0x39, 0x33, 0x42, 0x38, // 14C0FB80-F7C4-BB4B-1F24-496280B893B8

		0x01, // null

		0x04, // integer
		0x00, // 0

		0x04, // integer
		0x00, // 0

		0x09, 0x01, 0x01, // array, len 0
	}

	n := 0
	ret, err := ParseAMF3Val(data, &n)
	assertEqual(t, err, nil)
	assertEqual(t, ret.(AMFMap).Get("source").V, nil)
	assertEqual(t, ret.(AMFMap).Get("operation").V, "listAllPracticeGames")
	assertEqual(t, len(ret.(AMFMap).Get("headers").V.(AMFMap)), 3)
	assertEqual(t, ret.(AMFMap).Get("headers").V.(AMFMap).Get("DSRequestTimeout").V, 60)
	assertEqual(t, ret.(AMFMap).Get("headers").V.(AMFMap).Get("DSEndpoint").V, "my-rtmps")
	assertEqual(t, ret.(AMFMap).Get("headers").V.(AMFMap).Get("DSId").V, "BE241234-1E19-03C3-904C-2163660AB47F")
	assertEqual(t, ret.(AMFMap).Get("destination").V, "gameService")
	assertEqual(t, ret.(AMFMap).Get("messageId").V, "14C0FB80-F7C4-BB4B-1F24-496280B893B8")
	assertEqual(t, ret.(AMFMap).Get("clientId").V, nil)
	assertEqual(t, ret.(AMFMap).Get("timestamp").V, 0)
	assertEqual(t, ret.(AMFMap).Get("timeToLive").V, 0)
	assertEqual(t, len(ret.(AMFMap).Get("body").V.(AMFMap)), 0)
}

func TestReadString(t *testing.T) {
	data := []byte{
		0x06, 0x17, // string, len 11
		0x67, 0x61, 0x6d, 0x65, 0x53, 0x65, 0x72, 0x76, 0x69, 0x63, 0x65, // gameService
	}

	n := 0
	ret, err := ParseAMF3Val(data, &n)
	assertEqual(t, err, nil)
	assertEqual(t, ret.(string), "gameService")
}
